--[[

AutoMake
    by Oren Ben-Kiki (http://www.ben-kiki.org)
    copyright: GNU General Public License
    (http://www.gnu.org/copyleft/gpl.html)

Was initially based on Mat Perry's ReagentCalc add-on, which is no longer
maintained(?), but is now very different. Is _not_ based on Mat's
BetterTradeSkill add-on; queueing operations is done in a different (hopefully
more robust) way. I also peeked at BankStatement to see how to scan the bank
content and save the data between gaming sessions.

See ReadMe.txt for details.

--]]

-- Handles for integation into WarCraft UI.
local AM = {}
AutoMake = AM

-- Version of this code.
local CODE_VERSION = "1.2.5"

-- Colors (not all are used).
local COLOR_WHITE     = "|cffffffff"
local COLOR_RED       = "|cffff0000"
local COLOR_GREEN     = "|cff00ff00"
local COLOR_BLUE      = "|cff0000ff"
local COLOR_PURPLE    = "|cff700090"
local COLOR_YELLOW    = "|cffffff00"
local COLOR_ORANGE    = "|cffff6d00"
local COLOR_GREY      = "|cff808080"
local COLOR_GOLD      = "|cffcfb52b"
local COLOR_NEON_BLUE = "|cff4d4dff"
local COLOR_PINK      = "|cffffaa66"
local COLOR_END       = "|r"

-- Note this is the version of the data, not of the code.
local BANK_DATA_VERSION = 1
local SKILLS_DATA_VERSION = 1

-- Types of chat messages.
local MESG = 0
local INFO = 1
local WARN = 2
local BUGS = 3

-- Standard initialization.
function AM.OnLoad()

    AM.chatCount = 0
    AM.Chat(INFO, "Loading version " .. CODE_VERSION .. ".")

    -- Initialize local data.
    AM.tops = {}
    AM.skills = {}
    AM.reagents = {}
    AM.storeId = GetCVar("realmName") .. "|" .. UnitName("player")
    AM.inRescan = false
    AM.Dont()

    -- Set frame title.
    AutoMakeTitleText:SetText("AutoMake")

    -- Initialize inventory tracking.
    AM.ScanBags()

    -- Events for persistant data.
    this:RegisterEvent("VARIABLES_LOADED")

    -- Events for detecting production.
    this:RegisterEvent("CHAT_MSG_SPELL_TRADESKILLS")
    this:RegisterEvent("SPELLCAST_INTERRUPTED")
    this:RegisterEvent("SPELLCAST_FAILED")

    -- Events for tracking trade skills.
    this:RegisterEvent("TRADE_SKILL_SHOW")
    this:RegisterEvent("TRADE_SKILL_UPDATE")
    this:RegisterEvent("TRADE_SKILL_CLOSE")

    -- Events for tracking inventory.
    this:RegisterEvent("BAG_UPDATE")
    this:RegisterEvent("UNIT_INVENTORY_CHANGED")
    this:RegisterEvent("PLAYERBANKSLOTS_CHANGED")
    this:RegisterEvent("BANKFRAME_OPENED")

    -- Let user know we are here.
    AM.Message("AutoMake " .. CODE_VERSION .. " Loaded.")
end

-- Handle all sort of events.
function AM.OnEvent()
    AM.NonEmptyString("OnEvent", "event", event)

    -- Preserve for logging if it gets reset.
    local eventName = event .. ""

    AM.Chat(INFO, "OnEvent {" .. eventName .. "}, "
               .. "trade <" .. GetTradeSkillLine() .. ">, "
               .. "isDoing [" .. AM.isDoing .. "]")

    -- The saved variables have been loaded. Initialize persistant data.
    if (eventName == "VARIABLES_LOADED") then
        AM.Chat(INFO, "Fetch persistant data.")
        AM.FetchData()
        return
    end

    -- A new trade skill window has opened.
    if (eventName == "TRADE_SKILL_SHOW") then
        AM.Chat(INFO, "New window <" .. GetTradeSkillLine()
                   .. "> has opened: Scan, refresh, abort.")
        AM.ScanTrade()
        AM.Refresh()
        AM.Dont()
        return
    end

    -- The trade skill window has updated.
    if (eventName == "TRADE_SKILL_UPDATE") then
        AM.Chat(INFO, "New window <" .. GetTradeSkillLine()
                   .. "> was updated: Scan, refresh.")
        AM.ScanTrade()
        AM.Refresh()
        return
    end

    -- The trade skill window has been closed. No need to scan and refresh
    -- (there's no accessible data), but abort any ongoing processing.
    if (eventName == "TRADE_SKILL_CLOSE") then
        AM.Chat(INFO, "Old window <" .. GetTradeSkillLine()
                   .. "> has closed: Abort.")
        AM.Dont()
        return
    end

    -- Events indicating potential change to bank inventory.
    if (eventName == "PLAYERBANKSLOTS_CHANGED"
     or eventName == "BANKFRAME_OPENED") then
	if (BankFrame:IsVisible()) then
            AM.Chat(INFO, "Potential bank inventory change: Scan, refresh.")

            -- Rescan both bank and inventory, just to be on the safe side.
            AM.ScanBags()
            AM.ScanBank()
            AM.Refresh()
        end
        return
    end

    -- Events indicating potential change to inventory.
    if (eventName == "BAG_UPDATE"
     or eventName == "UNIT_INVENTORY_CHANGED"
     or eventName == "CHAT_MSG_SPELL_TRADESKILLS") then
        AM.Chat(INFO, "Potential bags inventory change: Scan.")

        -- Rescan inventory and bank if opened to detect updated amounts.
        AM.ScanBags()
	if (BankFrame:IsVisible()) then
            AM.ScanBank()
        end

        -- We may not be doing anything.
        if (AM.isDoing == "") then
            AM.Chat(INFO, "... Manual inventory change: Refresh.")
            AM.Refresh()

        -- It seems we are getting several BAG_UPDATE events when a new item is
        -- created. Only consider the one where the amount of the item has
        -- changed. The above "Refresh" has recomputed everything, so we are
        -- good to go to the next processing step.
        elseif (AM.hadBefore == AM.GetBagsAmount(AM.isDoing)) then
            AM.Chat(INFO, "... Had " .. AM.isDoing
                       .. " before " .. AM.hadBefore
                       .. ", have same now, why are we here? Refresh.")
            AM.Refresh()

        -- This happens when the trade skill window fails to come up.
        else
            AM.Chat(INFO, "... Had " .. AM.isDoing
                       .. " before " .. AM.hadBefore
                       .. ", have now " .. AM.GetBagsAmount(AM.isDoing)
                       .. ": Reduce, refresh, move to next.")
            AM.PositiveInt("OnEvent", "increase in amount of " .. AM.isDoing,
                           AM.GetBagsAmount(AM.isDoing) - AM.hadBefore)

            -- Reduce "todo" list if what we made was in it. Always subtract
            -- only 1 from "todo" list, because we do one thing at a time and
            -- we track "todo" by invocations, not amounts, to defuse issues
            -- of (random) multiple items created.
            AM.RemoveTop(AM.isDoing, 1)
            AM.Refresh()
            AM.Do(false)
        end

        return
    end

    -- If interrupted, stop doing whatever it is. This allows the user to abort
    -- the sequence. Usually all it takes is for the user to move.
    if (eventName == "SPELLCAST_INTERRUPTED") then
        if (AM.isDoing ~= "") then
            AM.Chat(INFO, "Spell we tried was interrupted: Abort.")
            AM.Message("Aborting.")
            AM.Dont()
        else
            AM.Chat(INFO, "Spell we didn't do was interrupted: Ignore.")
        end
        return
    end

    -- If failed, however, we can continue trying to do something else.
    if (eventName == "SPELLCAST_FAILED") then
        if (AM.isDoing ~= "") then
            AM.Chat(INFO, "Spell [" .. AM.isDoing
                       .. "] we tried has failed: Move to next.")

            -- Remember this skill won't work. We won't try it again unless we
            -- reset.
            AM.skillsWeFailed[AM.isDoing] = 1
            AM.Do(false)
        else
            AM.Chat(INFO, "Spell we didn't do has failed: Ignore.")
        end
        return
    end

    -- "Never happens".
    AM.Chat(BUGS, "Event {" .. eventName .. "} was not handled? Ignore.")
end

-- User asked to toggle the AutoMake frame.
function AM.Toggle()

    AM.Chat(INFO, "User hit toggle: Abort.")
    AM.Dont()

    -- Hide if shown.
    if (AutoMakeFrame:IsVisible()) then
        AM.Chat(INFO, "Is visible: Hide.")
        AutoMakeFrame:Hide()
        AutoMakeButtonsFrame:Hide()

    -- Show if hidden. Refresh just in case.
    else
        AM.Chat(INFO, "Is hidden: Show, refresh.")
        AutoMakeFrame:Show()
        AutoMakeButtonsFrame:Show()
        AM.Refresh()
    end
end

-- User asked to clear the "todo" list.
-- Also abort any on-going processing.
function AM.Clear()

    AM.Chat(INFO, "User hit clear: Abort.")
    AM.Dont()

    -- Clear everything except for the persistant storage.
    AM.tops = {}
    AM.skills = {}
    AM.reagents = {}

    -- Clear displayed text.
    AM.UpdateText()
end

-- Recompute evrything and display updated data. Called all over the place.
function AM.Refresh()

    AM.Chat(INFO, "Refreshing...")

    -- Remember the current "todo" list.
    local oldTops = AM.tops

    -- Clear everything except for the persistant storage.
    AM.tops = {}
    AM.skills = {}
    AM.reagents = {}

    -- Add the "todo" list again. This causes recomputing everything according
    -- to the current inventory and bank statement.
    for name, info in pairs(oldTops) do
        AM.AddTop(info.skill, name, info.link, info.invokes, false, info.trade)
    end

    -- Update displayed text accordingly.
    AM.UpdateText()
end

-- User asked to remove the current item from the "todo" list.
function AM.Remove(isAll)
    AM.NonNilBool("Remove", "isAll", isAll)

    if (isAll) then
        AM.Chat(INFO, "User hit Remove All. Abort.")
    else
        AM.Chat(INFO, "User hit Remove. Abort.")
    end
    AM.Dont()

    -- What to remove from list.
    local skillId, repeatInvokes = AM.CurrentSelection()
    if (not skillId or skillId < 0) then
        return
    end
    local name, _ = AM.GetSkillItemName(skillId)

    -- How much to remove from list.
    if (isAll and AM.tops[name]) then
        repeatInvokes = AM.tops[name].invokes
    end

    -- Remove from list and refresh display accordingly.
    AM.RemoveTop(name, repeatInvokes)
    AM.Refresh()
end

-- Remove a specified number of items from the "todo" list. Requires calling
-- refresh afterwards to get rid of required reagents and intermediate steps.
function AM.RemoveTop(name, repeatInvokes)
    AM.NonEmptyString("RemoveTop", "name", name)
    AM.PositiveInt("RemoveTop", "repeatInvokes", repeatInvokes)

    -- The user may hit remove for an item not in the list. Also we might get
    -- production events for items not in the list.
    if (not AM.tops[name]) then
        AM.Chat(INFO, "Remove " .. repeatInvokes
                   .. " x [" .. name .. "] non-top, ignore.")
        return
    end

    -- We might not need any of the items at all. Completely remove list entry.
    if (AM.tops[name].invokes <= repeatInvokes) then
        AM.Chat(INFO, "Remove " .. repeatInvokes
                   .. " x [" .. name .. "] top completely out of "
                   .. AM.tops[name].invokes .. ".")
        AM.tops[name] = nil
        return
    end

    -- Otherwise just reduce the needed amount. List entry persists.
    AM.Chat(INFO, "Remove " .. repeatInvokes
               .. " x [" .. name .. "] top partially out of "
               .. AM.tops[name].invokes .. ".")
    AM.tops[name].invokes = AM.tops[name].invokes - repeatInvokes
end

-- Add the current item to the "todo" list.
function AM.Add(isAll)
    AM.NonNilBool("Add", "isAll", isAll)

    if (isAll) then
        AM.Chat(INFO, "User hit Add All. Abort.")
    else
        AM.Chat(INFO, "User hit Add. Abort.")
    end
    AM.Dont()

    -- What to add to list.
    local skillId, repeatInvokes = AM.CurrentSelection()
    if (skillId < 0) then
        return
    end
    local name, link = AM.GetSkillItemName(skillId)

    -- Add to list. This updates all intermediate steps etc.
    AM.AddTop(skillId, name, link, repeatInvokes, isAll, GetTradeSkillLine())

    -- Update displayed text accordingly. Note here we don't refresh; rather,
    -- refresh calls this function to work.
    AM.UpdateText()
end

-- Add a specified number of items to the "todo" list, and also all their
-- dependencies and intermediate steps.
function AM.AddTop(skillId, name, itemLink, repeatInvokes, isAll, tradeSkill)
    AM.NonNegativeInt("AddTop", "skillId", skillId)
    AM.NonEmptyString("AddTop", "name", name)
    AM.NonEmptyString("AddTop", "itemLink", itemLink)
    AM.PositiveInt("AddTop", "repeatInvokes", repeatInvokes)
    AM.NonNilBool("AddTop", "isAll", isAll)
    AM.NonEmptyString("AddTop", "tradeSkill", tradeSkill)

    -- If asked for all, only add what we need to get to it.
    if (isAll) then

        -- For code simplicity - remove all the current invocations first.
        if (AM.tops[name]) then
            AM.RemoveTop(name, AM.tops[name].invokes)
            AM.Refresh()
        end

        -- Find out how many we can make and try to make that, regerdless of
        -- the number displayed in the trade skill window.
        local _, _, available, _ = GetTradeSkillInfo(skillId)
        AM.NonNilInt("GetTradeSkillInfo(" .. skillId .. ")",
                     "returned available", available)
        if (available <= 0) then
            return
        end
        repeatInvokes = available
    end

    -- Initialize or update the needed top-level invocations.
    if (not AM.tops[name]) then
        AM.tops[name] = {
            trade = tradeSkill,
            skill = skillId,
            link = itemLink,
            invokes = 0
        }
    end
    AM.tops[name].invokes = AM.tops[name].invokes + repeatInvokes

    -- Initialize tracking the skill required to produce the item. Here
    -- amounts are in actual items, not invocations.
    if (not AM.skills[name]) then
        AM.skills[name] = {
            trade = tradeSkill,
            skill = skillId,
            link = itemLink,
            needTop = 0,    -- Due to AddTop commands.
            needOld = 0,    -- Used out of inventory.
            needNew = 0,    -- Need to be created.
            extraNew = 0    -- Produced but not used.
        }
    end

    -- Recursively add the skill and anything it needs, taking into account
    -- that a single invocation may create more than one item.
    local numMade, _ = AM.GetSkillNumMade(name)
    AM.RecursiveAdd(name, itemLink, repeatInvokes * numMade, true)
end

-- Recursively add a number of items (and everything they depend on). Needs to
-- know whether this is a top ("todo" list) item rather than an intermediate
-- because "todo" list items must not be provided from the inventory.
function AM.RecursiveAdd(name, link, needItems, isNeedTop)
    AM.NonEmptyString("AddTop", "name", name)
    AM.PositiveInt("AddTop", "needItems", needItems)
    AM.NonNilBool("AddTop", "isNeedTop", isNeedTop)

    if (isNeedTop) then
        AM.Chat(INFO, "Add top " .. needItems .. " x [" .. name .. "]")
    else
        AM.Chat(INFO,
                "Add intermediate " .. needItems .. " x [" .. name .. "]")
    end

    -- Only intermediate reagents may be taken from inventory.
    if (not isNeedTop) then

        -- How many existing items can we use?
        local canUseExisting

        -- If we already decided to create some, perhaps we made some extra.
        if (AM.skills[name].needNew > 0) then
            canUseExisting = min(AM.skills[name].extraNew, needItems)
            AM.skills[name].extraNew =
                                    AM.skills[name].extraNew - canUseExisting
            AM.Chat(INFO, "Use " .. canUseExisting .. " extra.")

        -- We didn't, we might have some available old ones in the inventory.
        else
            canUseExisting = AM.GetBagsAmount(name) - AM.skills[name].needOld
            canUseExisting = min(canUseExisting, needItems)
            AM.skills[name].needOld = AM.skills[name].needOld + canUseExisting
            AM.Chat(INFO, "Use " .. canUseExisting .. " old.")
        end

        -- If existing items are enough, no need to look any further.
        needItems = needItems - canUseExisting
        AM.Chat(INFO, "Need " .. needItems .. " now.")
        if (needItems == 0) then
            return
        end
    end

    -- Did this came from "Add"?
    if (isNeedTop) then
        AM.skills[name].needTop = AM.skills[name].needTop + needItems
        AM.Chat(INFO, "Need " .. AM.skills[name].needTop .. " top now.")

    -- Or from a recursive call?
    else
        AM.skills[name].needNew = AM.skills[name].needNew + needItems
        AM.Chat(INFO, "Need " .. AM.skills[name].needNew .. " new now.")
    end

    -- How many invocations will be required?
    local numMade, _ = AM.GetSkillNumMade(name)
    local needInvokes = ceil(needItems * 1.0 / numMade)
    AM.Chat(INFO, "Need " .. needInvokes .. " invokes")

    -- This might result in us creating extra items.
    local extraNew = needInvokes * numMade - needItems
    AM.skills[name].extraNew = AM.skills[name].extraNew + extraNew
    AM.Chat(INFO, "Will make " .. AM.skills[name].extraNew .. " extra.")

    -- Consider reagents required.
    for reagentName, reagentData
     in pairs(AutoMake_TradeSkillInfo[AM.storeId][name].reagents) do

        -- How many of it do we need.
        local needReagents = reagentData.neededItems * needInvokes

        -- Use the link for display - it color codes it by rarity.
        local reagentLink = reagentData.link

        -- Test whether we can make this using an intermediate step.
        local reagentInfo = AutoMake_TradeSkillInfo[AM.storeId][reagentName]
        if (reagentInfo) then

            AM.Chat(INFO, "Require " .. needReagents
                       .. " x [" .. reagentName .. "] intermediate")

            -- We might not have known link of top level items;
            -- Fill it from the trade skill scanning.
            if (not reagentLink) then
                reagentLink = reagentInfo.link
            end

            -- Initialize the reagent's skill if not seen before.
            if (not AM.skills[reagentName]) then
                AM.skills[reagentName] = {
                    trade = reagentInfo.trade,
                    skill = reagentInfo.skill,
                    link = reagentLink,
                    needOld = 0,
                    needNew = 0,
                    needTop = 0,
                    extraNew = 0
                }
            end

            -- Recursively add the reagent's skill required reagents, etc.
            AM.RecursiveAdd(reagentName, reagentLink, needReagents, false)

        -- We can't make this using an intermediate step. Add it as a reagent.
        else

            AM.Chat(INFO, "Require " .. needReagents
                       .. " x [" .. reagentName .. "] reagents")

            -- Initialize reagent's data.
            if (not AM.reagents[reagentName]) then
                AM.reagents[reagentName] = {
                    need = needReagents,
                    link = reagentLink
                }

            -- Or update the reagent's data.
            else
                AM.reagents[reagentName].need =
                                   AM.reagents[reagentName].need + needReagents
            end
        end
    end

    AM.Chat(INFO, "Added all reagents for " .. name)
end

-- Obtain the currently selected skill and invocations count.
function AM.CurrentSelection()

    -- Find out yhe selected skill.
    local skillId = GetTradeSkillSelectionIndex()
    if (not skillId or skillId < 0) then
        AM.Message("You need to select something in the Trade Skill window.")
        return -1, -1
    end

    -- Find out how many invocations are required.
    local invokes = TradeSkillInputBox:GetNumber()
    AM.PositiveInt("TradeSkillInputBox:GetNumber()", "returned", invokes)

    -- Return pair.
    return skillId, invokes
end

-- Add all "basic" steps to the "todo" list.
function AM.Basics()

    AM.Chat(INFO, "User hit Basics. Abort.")
    AM.Dont()

    -- Loop on all the trade skills and check them.
    local numSkills = GetNumTradeSkills()
    for skillId = 1, numSkills do

        -- Can we make any?
        local skillName, itemLink = AM.GetSkillItemName(skillId)
        local _, _, available, _ = GetTradeSkillInfo(skillId)
        AM.NonNilInt("GetTradeSkillInfo(" .. skillId .. ")",
                     "returned available", available)
        if (available <= 0) then
            AM.Chat(INFO, "Can't make any [" .. skillName .. "]")

        -- We can make some.
        else
            AM.Chat(INFO,
                    "Can make " .. available .. " x [" .. skillName .. "]")

            -- Assume it is basic and try to prove otherwise.
            local isBasic = true

            -- Examine all reagents to find out.
            for reagentName
             in AutoMake_TradeSkillInfo[AM.storeId][skillName].reagents do

                -- If we can make the reagent, the skill isn't basic.
                if (AutoMake_TradeSkillInfo[AM.storeId][reagentName]) then
                    AM.Chat(INFO, "Can make reagent [" .. reagentName
                               .. "]. [" .. skillName .. "] is not basic.")
                    isBasic = false
                    break

                -- If the reagent is used by other skills, this isn't basic.
                elseif (not AM.UsedOnlyBy(reagentName, skillName)) then
                    AM.Chat(INFO, "Hence [" .. skillName .. "] is not basic.")
                    isBasic = false

                -- Can make it, and only one using it - may be basic.
                else
                    AM.Chat(INFO, "Hence [" .. skillName .. "] may be basic.")
                end
            end

            -- If it is basic, add it to the list.
            if (isBasic) then
                AM.Chat(INFO, "[" .. skillName .. "] turned out to be basic.")

                -- Add all we can make to the "todo" list.
                AM.AddTop(skillId, skillName, itemLink, 1, true,
                          GetTradeSkillLine())
            end
        end
    end

    -- Recompute everything.
    AM.Refresh()
end

-- Test whether a reagent is used only by one skill.
function AM.UsedOnlyBy(reagentName, skillName)
    AM.NonEmptyString("UsedOnlyBy", "reagentName", reagentName)
    AM.NonEmptyString("UsedOnlyBy", "skillName", skillName)

    -- Loop on all skills we know. Note this includes skills from other
    -- trades.
    for otherSkill, otherInfo in pairs(AutoMake_TradeSkillInfo[AM.storeId]) do

        -- Skip the one we already know about.
        if (otherSkill ~= skillName) then
            for otherReagent in otherInfo.reagents do
                if (otherReagent == reagentName) then
                    AM.Chat(INFO, "Reagent [" .. reagentName
                               .. "] is used by both [" .. skillName
                               .. "] and [" .. otherSkill .. "]")
                    return false
                end
            end
        end
    end

    -- No match found, so is used only by the one skill.
    AM.Chat(INFO, "Reagent [" .. reagentName
               .. "] is used only by [" .. skillName .. "]")
    return true
end

-- Display the updated information in the window.
function AM.UpdateText()

    -- Total text to be displayed in AutoMake frame, built incrementally.
    local text = AM.TopsText() .. AM.StepsText() .. AM.ReagentsText()

    -- If none of the above yielded anything, there's nothing to create.
    if (text == "") then
        text = "No items to create.\n"
    end

    -- Update frame's text.
    AutoMakeInfoText:SetText(text)
end

-- Text for the top ("todo") list items.
function AM.TopsText()

    -- Loop on all skills and look for needTop.
    local text = ""
    for name, info in AM.SortedPairs(AM.skills) do
        if (info.needTop > 0) then

            -- Find out how much we have.
            local haveInBags = AM.GetBagsAmount(name)
            local haveInBank = AM.GetBankAmount(name)

            -- Here we never need anything (we are always making), so turn off
            -- coloring by using -1.
            text = text .. " "
                .. AM.TopNeedText(name, 0) .. " "
                .. AM.LinkOrName(info.link, name) .. " "
                .. AM.TradeText(info.trade) .. " "
                .. AM.AmountsText(haveInBags, -1, haveInBank) .. "\n"
        end
    end

    -- Add header line only if there are any top ("todo") items.
    if (text ~= "") then
        text = "In order to create:\n" .. text .. "\n"
    end
    return text
end

-- Text for the intermediate steps.
function AM.StepsText()

    -- Loop on all skills and look for needNew.
    local text = ""
    for name, info in AM.SortedPairs(AM.skills) do
        if (info.needNew > 0) then

            -- Here the current inventory is irrelevant; if there are any,
            -- they are all used as reagents. Otherwise, we wouldn't have
            -- marked this as an intermediate step. Likewise, there's no need
            -- for "need X more", because we are making exactly that. Don't
            -- color the amount here either.
            local haveInBank = AM.GetBankAmount(name)
            text = text .. "  "
                .. info.needNew .. " "
                .. AM.LinkOrName(info.link, name) .. " "
                .. AM.TradeText(info.trade) .. " "
                .. AM.AmountsText(0, -1, haveInBank) .. "\n"
        end
    end

    -- Add header line only if there are any intermediate steps.
    if (text ~= "") then
        text = "Using steps:\n" .. text .. "\n"
    end
    return text
end

-- Text for the reagents needed for the intermediate steps and the "todo" list.
function AM.ReagentsText()
    -- There are two types of reagents we list.
    local reagents = AM.CanMakeText() .. AM.CantMakeText()
    -- Add header line only if there are any intermediate steps.
    if (reagents ~= "") then
        reagents = "You will need:\n" .. reagents .. "\n"
    end
    return reagents
end

-- Text for the reagent we can produce, but can use from inventory.
function AM.CanMakeText()

    -- Loop on all skills and look for needOld.
    local text = ""
    for name, info in AM.SortedPairs(AM.skills) do
        if (info.needOld > 0) then

            -- Here we always use from inventory, so there's no "need X more".
            -- Any such are counted as intermediate steps above. Hence the
            -- amount is always displayed in green.
            local haveInBags = AM.GetBagsAmount(name)
            local haveInBank = AM.GetBankAmount(name)
            text = text .. "  "
                .. AM.Have(info.needOld) .. " "
                .. AM.LinkOrName(info.link, name) .. " "
                .. AM.AmountsText(haveInBags, 0, haveInBank) .. "\n"
        end
    end
    return text
end

-- Text for reagents we can't produce.
function AM.CantMakeText()

    -- Loop on all reagents. Here we use colors to show what is available
    -- (green), missing (red), and missing but can be fetched from the bank
    -- (orange).
    local text = ""
    for name, info in AM.SortedPairs(AM.reagents) do

        -- Here finally all three fields make sense.
        local haveInBags = AM.GetBagsAmount(name)
        local haveInBank = AM.GetBankAmount(name)
        local needMore = info.need - haveInBags

        -- Handle the text prefix first.
        if (needMore <= 0) then
            text = text .. "  " .. AM.Have(info.need)
            needMore = 0
        elseif (needMore <= haveInBank) then
            text = text .. "  " .. AM.HaveInBank(info.need)
        else
            text = text .. "  " .. AM.NeedMore(info.need)
        end

        -- Then the normal amounts part.
        text = text .. " "
            .. AM.LinkOrName(info.link, name) .. " "
            .. AM.AmountsText(haveInBags, needMore, haveInBank) .. "\n"
    end
    return text
end

-- Return the text to display for required amount of a top ("todo") list item.
function AM.TopNeedText(name, extra)
    AM.NonEmptyString("TopNeedText", "name", name)
    AM.NonNegativeInt("TopNeedText", "extra", extra)

    local invokes = AM.tops[name].invokes
    local minMade, maxMade = AM.GetSkillNumMade(name)
    if (minMade == maxMade) then
        return invokes * minMade + extra
    else
        return (invokes * minMade + extra) .. "-"
            .. (invokes * maxMade + extra)
    end
end

-- Return the item link if available, otherwise the item name.
function AM.LinkOrName(link, name)
    AM.NonEmptyString("LinkOrName", "name", name)
    if (link) then
        return link
    else
        return name
    end
end

-- Amounts text with surrounding () and colors as needed.
function AM.AmountsText(haveInBags, needMore, haveInBank)
    AM.NonNegativeInt("AmountsText", "haveInBags", haveInBags)
    AM.NonNilInt("AmountsText", "needMore", needMore)
    AM.NonNegativeInt("AmountsText", "haveInBank", haveInBank)

    -- Text is built incrementally.
    local text = ""

    -- First how much we have (in the bags), in green if we have enough.
    if (haveInBags > 0) then
        text = "have " .. haveInBags
        if (needMore == 0) then
            text = AM.Have(text)
        end
    end

    -- Then how much more we need, in red if we need more.
    if (needMore > 0) then
        if (text ~= "") then
            text = text .. ", "
        end
        text = text .. AM.NeedMore("need " .. needMore .. " more")
    end

    -- Then how much we have in the bank.
    if (haveInBank > 0) then
        if (text ~= "") then
            text = text .. ", "
        end

        -- In Orange if we have enough, otherwise normal.
        if (haveInBank >= needMore and needMore > 0) then
            text = text .. AM.HaveInBank(haveInBank .. " in bank")
        else
            text = text .. haveInBank .. " in bank"
        end
    end

    -- Wrap all the above in () only if non-empty.
    if (text ~= "") then
        text = "(" .. text .. ")"
    end

    return text
end

-- Return a color coded text for the trade needed for a step.
function AM.TradeText(trade)
    AM.NonEmptyString("TradeText", "trade", trade)

    if (trade == GetTradeSkillLine()) then
        return AM.Have("<" .. trade .. ">")
    else
        return AM.HaveInBank("<" .. trade .. ">")
    end
end

-- Text colored for reagents user has enough of in his bags.
function AM.Have(text)
    AM.NonEmptyString("Have", "text", text)

    return COLOR_GREEN .. text .. COLOR_END
end

-- Text colored for reagents user have enough of in the bank.
function AM.HaveInBank(text)
    AM.NonEmptyString("HaveInBank", "text", text)

    return COLOR_ORANGE .. text .. COLOR_END
end

-- Text colored for reagents user need more of.
function AM.NeedMore(text)
    AM.NonEmptyString("NeedMore", "text", text)

    return COLOR_RED .. text .. COLOR_END
end

-- Do the next operation (will cause all remaining ones to be done via events).
function AM.Do(toReset)
    AM.NonNilBool("Do", "toReset", toReset)

    -- Called from GUI? Start a fresh build.
    if (toReset) then
        AM.Chat(INFO, "Restarting using <" .. GetTradeSkillLine() .. ">")
        AM.Dont()

    -- Called from event handler, move to next step.
    else
        AM.Chat(INFO, "Continuing using <" .. GetTradeSkillLine() .. ">")

        -- Until proven otherwise.
        AM.isDoing = ""
        AM.hadBefore = 0
    end

    -- Need steps from other trade skills.
    local otherTrades = {}

    -- Have we tried and failed some skills?
    local failedSkills = false

    -- Do we have skills that are missing reagents?
    local missingSkills = false

    -- Refresh EVERYTHING so we KNOW all amounts etc. are updated. This is
    -- purely defensive, as event handling should have kept everything updated
    -- anyway. Better safe than sorry...
    AM.ScanBags()
    if (BankFrame:IsVisible()) then
        AM.ScanBank()
    end
    AM.ScanTrade()
    AM.Refresh()

    -- The current trade skill.
    local currentTrade = GetTradeSkillLine()
    AM.NonEmptyString("GetTradeSkillLine()", "returned trade", currentTrade)

    -- Look for something we can do.
    for name, info in AM.SortedPairs(AM.skills) do

        -- Check if we need to make this at all.
        local need = info.needTop + info.needNew
        if (need > 0) then

            -- Do we have the wrong trade skill window opened?
            if (info.trade ~= currentTrade) then
                otherTrades[info.trade] = 1

                AM.Chat(INFO, "Can't do [" .. name
                           .. "] because it requires <"
                           .. info.trade .. "> and we are using <"
                           .. GetTradeSkillLine() .. ">")

            -- We do have the right trade skill window opened. But maybe this
            -- skill was tried and failed due to missing requirements.
            elseif (AM.skillsWeFailed[name]) then
                failedSkills = true

                AM.Chat(INFO, "Won't try to do [" .. name
                           .. "] because it failed before")

            -- Nope, first time we are trying it.
            else

                -- Do we have the reagents for the skill?
                local _, _, available, _ = GetTradeSkillInfo(info.skill)
                AM.NonNilInt("GetTradeSkillInfo(" .. info.skill .. ")",
                             "returned available", available)
                if (available <= 0) then
                    missingSkills = true

                    AM.Chat(INFO, "Can't do [" .. name
                               .. "] because it is missing reagents")

                -- The nice thing about simply recomputing everything when the
                -- inventory changes is that we can simply invoke each skill
                -- once at a time and all dependencies and amounts are
                -- automatically taken care of.
                else
                    AM.isDoing = name
                    AM.hadBefore = AM.GetBagsAmount(name)
                    AM.Chat(INFO, "Had " .. AM.hadBefore
                               .. " before doing [" .. name .. "]")
                    local needText
                    if (AM.tops[name]) then
                        needText = AM.TopNeedText(name, info.needNew)
                    else
                        needText = need
                    end
                    AM.Message("Making " .. needText .. " x [" .. name .. "]")
                    DoTradeSkill(info.skill, 1)
                    return
                end
            end
        end
    end

    -- Tell the user why we have stopped.
    AM.Chat(INFO, "Tell user why we are done.")

    -- Perhaps we failed doing some skills (missing anvil, forge, etc.).
    if (failedSkills) then
        AM.Message("Missing requirements.")
        AM.Dont()
        return
    end

    -- Perhaps we need skills from another trade. It would be very nice to be
    -- able to switch to the other trades and continue. This requires
    -- memorizing which trade skills we tried and can't advance in, to avoid
    -- looping. This list needs to be reset every time we manage to create
    -- something. There's no point in maintaining this list, because it is
    -- impossible to switch to another trade skill and continue, probably
    -- because of Blizzard's anti-bot policy.
    local otherTradesText = AM.Join(otherTrades, " and ")
    if (otherTradesText ~= "") then
        AM.Message("In " .. currentTrade .. ", need " .. otherTradesText .. ".")
        AM.Dont()
        return
    end

    -- Perhaps we are missing some reagents.
    if (missingSkills) then
        AM.Message("Missing reagents.")
        AM.Dont()
        return
    end

    -- If none of the above, then we are simply done.
    AM.Message("Done.")
    AM.Dont()
end

-- Stop doing anything, clear everything.
function AM.Dont()
    AM.Chat(INFO, "Stop doing.")
    AM.isDoing = ""
    AM.hadBefore = 0
    AM.skillsWeFailed = {}
end

-- Utility to convert the index of a trade skill (under the current, open,
-- trade skill window) into a usable item name and a color-coded link.
function AM.GetSkillItemName(skillId)
    AM.NonNegativeInt("GetSkillItemName", "skillId", skillId)

    -- This translates things like "Smelt Copper" to "Copper Ore".
    local link = GetTradeSkillItemLink(skillId)
    if (link) then
        local _, _, item = string.find(link, "^.*%[(.*)%].*$")
        return item, link
    else
        local item, _, _ = GetTradeSkillInfo(skillId)
        AM.NonEmptyString("GetTradeSkillInfo(" .. skillId .. ")",
                          "returned item", item)
        return item, nil
    end
end

-- Scan all skills in the current trade skill window.
function AM.ScanTrade()
    local retry = 1
    while (retry > 0) do
        retry = AM.RetryScanTrade(retry)
    end
end

-- Retry scanning all skills in the current trade skill window.
function AM.RetryScanTrade(retry)

    -- "Never happens".
    local tradeName = GetTradeSkillLine()
    AM.NonEmptyString("GetTradeSkillLine()", "returned trade", tradeName)
    if (tradeName == "UNKNOWN") then
        AM.inRescan = false
        AM.Chat(WARN, "ScanTrade called when trade skill window is closed.")
        return 0
    end

    -- Loop on all the trade skills and update them.
    local numSkills = GetNumTradeSkills()
    for skillId = 1, numSkills do

        -- Test this isn't just a title line.
        local skillName, itemLink = AM.GetSkillItemName(skillId)
        if (itemLink) then

            -- Skill id might have changed due to learning new skills.
            if (AutoMake_TradeSkillInfo[AM.storeId][skillName]) then
                AutoMake_TradeSkillInfo[AM.storeId][skillName].skill = skillId

            -- New skill, fill in all information.
            else
                -- Initialize data with how many are made.
                local minMade, maxMade = GetTradeSkillNumMade(skillId)
                AM.PositiveInt("GetTradeSkillNumMade(" .. skillId .. ")",
                               "returned min made", minMade)
                AM.PositiveInt("GetTradeSkillNumMade(" .. skillId .. ")",
                               "returned max made", maxMade)
                AutoMake_TradeSkillInfo[AM.storeId][skillName] = {
                    trade = tradeName,
                    skill = skillId,
                    link = itemLink,
                    min = minMade,
                    max = maxMade,
                    reagents = {},
                }

                -- Find out which reagents it needs.
                local numReagents = GetTradeSkillNumReagents(skillId)
                AM.NonNegativeInt("GetTradeSkillNumReagents(" .. skillId .. ")",
                                  "returned reagents number", numReagents)
                for reagentId = 1, numReagents do
                    local reagentName, _, reagentCount, _ =
                                   GetTradeSkillReagentInfo(skillId, reagentId)
                    -- This should never happen, yet users reported it. Try
                    -- and recover by re-scanning the list, it sometimes helps.
                    -- If we completely fail, close the trade skill window;
                    -- this will force the user to re-open it and hopefully a
                    -- rescan after a close/open will finally succeed. There is
                    -- always the chance this will stop people from
                    if (not reagentName or not reagentCount) then
                        AM.Chat(BUGS, "GetTradeSkillReagentInfo<" .. tradeName
                                   .. ">(" .. skillId .. "=[" .. skillName
                                   .. "], " .. reagentId .. "/" .. numReagents
                                   .. ") attempt " .. retry
                                   .. "/3 returned nil(s).")
                        -- First, retry 3 times.
                        if (retry < 3) then
                            return retry + 1

                        -- Then, try closing the trade skill window and let
                        -- the user re-open it.
                        elseif (not AM.inRescan) then
                            AM.Chat(BUGS, "Close trade skill window, maybe "
                                       .. "we'll have better luck on re-open")
                            AM.Message("Blizzard bug detected - re-open "
                                    .. tradeName .. " window to recover.")
                            AM.inRescan = true
                            CloseTradeSkill()
                        else
                            AM.Message("Recovery from Blizzard bug failed.")
                            AM.inRescan = false
                            AM.Chat(BUGS,
                                    "Neither retries nor close/reopen helped. "
                                 .. "Give up. Here comes the error pop-up...")
                        end
                    end
                    AM.NonEmptyString("GetTradeSkillReagentInfo(" .. skillId
                                   .. ", " .. reagentId .. ")",
                                      "reagent name", reagentName)
                    AM.PositiveInt("GetTradeSkillReagentInfo(" .. skillId
                                .. ", " .. reagentId .. ")",
                                   "reagent amount", reagentCount)
                    local reagentLink =
                               GetTradeSkillReagentItemLink(skillId, reagentId)
                    AutoMake_TradeSkillInfo[AM.storeId]
                                       [skillName].reagents[reagentName] = {
                        neededItems = reagentCount,
                        link = reagentLink
                    }
                end
            end
        end
    end
    AM.inRescan = false
    return 0
end

-- Fin d out how many items are made in single skill invocation.
function AM.GetSkillNumMade(skillName)
    AM.NonEmptyString("GetSkillNumMade", "skillName", skillName)
    if (not AutoMake_TradeSkillInfo[AM.storeId][skillName]) then
        AM.Chat(BUGS, "Asked for number made for unknown skill " .. skillName)
        return 1, 1
    else
        return AutoMake_TradeSkillInfo[AM.storeId][skillName].min,
               AutoMake_TradeSkillInfo[AM.storeId][skillName].max
    end
end

-- Scan all the bags in the inventory and update quantities accordingly.
function AM.ScanBags()
    -- Forget previous inventory.
    AM.bagItemQuantity = {}

    -- Scan the items worn by the character.
    AM.ScanInventory()

    -- Scan all the inventory bags (including what is worn, for stuff like
    -- goggles).
    for bagNum = 0, NUM_BAG_SLOTS do
        AM.ScanBag(AM.bagItemQuantity, bagNum)
    end
end

-- Scan all the slots in the inventory. It isn't _quite_ useless. For
-- example, some goggles are reagents of other goggles.
function AM.ScanInventory()

    -- Scan all slots. There is probably a global variable containing "23"...
    -- 23 is the last slot, containing ammo, we skip that because it is counted
    -- in the bags already.
    for slotNum = 0, 22 do

        -- Some slots are empty.
        local link = GetInventoryItemLink("player", slotNum)
        if (link) then
            local name = AM.NameOfLink(link)
            local quantity = GetInventoryItemCount("player", slotNum)
            AM.NonNegativeInt("GetInventoryItemCount('player', "
                           .. slotNum .. ")", "returned amount", quantity)
            if (not AM.bagItemQuantity[name]) then
                AM.bagItemQuantity[name] = quantity
            else
                AM.bagItemQuantity[name] = AM.bagItemQuantity[name] + quantity
            end
        end
    end
end

-- Scan all the bags in the bank and update quantities accordingly.
function AM.ScanBank()

    -- Forget previous inventory.
    AutoMake_BankItemQuantity[AM.storeId] = {}

    -- Scan the main bank container.
    AM.ScanBag(AutoMake_BankItemQuantity[AM.storeId], BANK_CONTAINER)

    -- Scan all the bags contained in the bank.
    for bagNum = NUM_BAG_SLOTS + 1, NUM_BAG_SLOTS + NUM_BANKBAGSLOTS do
        AM.ScanBag(AutoMake_BankItemQuantity[AM.storeId], bagNum)
    end
end

-- Scan one of the player's bank bags and update quantities accordingly.
function AM.ScanBag(container, bagNum)
    AM.NonNilTable("ScanBag", "container", container)
    AM.NonNilInt("ScanBag", "bagNum", bagNum)

    -- Not all bank bags actually exist.
    local numSlots = GetContainerNumSlots(bagNum)
    if (numSlots <= 0) then
        return
    end

    -- For those that do, count the quantity of all contained items.
    for slotNum = 1, numSlots do
        local name, quantity = AM.GetContainerItemInfo(bagNum, slotNum)
        if (name) then
            if (not container[name]) then
                container[name] = quantity
            else
                container[name] = container[name] + quantity
            end
	end
   end
end

-- Get the name and quantity of an item in the bank.
function AM.GetContainerItemInfo(bagNum, slotNum)
    AM.NonNilInt("ScanBag", "bagNum", bagNum)
    AM.NonNegativeInt("ScanBag", "slotNum", slotNum)

    -- If the slot is empty, the link will be nil.
    local link = GetContainerItemLink(bagNum, slotNum)
    if (not link) then
        return nil, nil
    end

    -- Slot isn't empty, compute and return data.
    local name = AM.NameOfLink(link)
    local _, quantity = GetContainerItemInfo(bagNum, slotNum)
    AM.PositiveInt("GetContainerItemInfo(" .. bagNum .. ", " .. slotNum .. ")",
                   "returned amount", quantity)
    return name, quantity
end

-- Convert a link to an item name, This has been lifted as-is from
-- BankStatement. The regexp can probably be optimized, but this works and I
-- don't have the strength to fiddle with it.
function AM.NameOfLink(link)
    AM.NonEmptyString("NameOfLink", "link", link)

    -- The regexp can probably be optimized, but this works and I don't have
    -- the strength to fiddle with it.
    for name
     in string.gfind(link, "|c%x+|Hitem:%d+:%d+:%d+:%d+|h%[(.-)%]|h|r") do
        return name
    end

    -- "Never happens".
    AM.Chat(BUGS, "Could not extract name from link " .. link)
    return link
end

-- Get the quantity of some item in the bank.
function AM.GetBankAmount(name)
    AM.NonEmptyString("GetBankAmount", "name", name)

    -- Ensure tables exist.
    if (not AutoMake_BankItemQuantity) then
        AutoMake_BankItemQuantity = {}
    end
    if (not AutoMake_BankItemQuantity[AM.storeId]) then
        AutoMake_BankItemQuantity[AM.storeId] = {}
    end

    -- Simply look it up.
    if (not AutoMake_BankItemQuantity[AM.storeId][name]) then
        return 0
    else
        return AutoMake_BankItemQuantity[AM.storeId][name]
    end
end

-- Get the quantity of some item in the inventory.
function AM.GetBagsAmount(name)
    AM.NonEmptyString("GetBagsAmount", "name", name)

    -- Simply look it up.
     if (not AM.bagItemQuantity[name]) then
        return 0
    else
        return AM.bagItemQuantity[name]
    end
end

-- The data is actually fetched automatically. This verifies the version is
-- compatible with what we expect.
function AM.FetchData()

    -- If there is no data, mark it as invalid versions.
    if (not AutoMake_DataVersion) then
        AutoMake_DataVersion = {
            bank = -1,
            skills = -1,
        }
    end

    -- Clear out-of-date bank inventory information.
    if (AutoMake_DataVersion.bank ~= BANK_DATA_VERSION) then
        AutoMake_BankItemQuantity = {}
        AutoMake_DataVersion.bank = BANK_DATA_VERSION
    end

    -- Clear out-of-date production skills information.
    if (AutoMake_DataVersion.skills ~= SKILLS_DATA_VERSION) then
        AutoMake_TradeSkillInfo = {}
        AutoMake_DataVersion.skills = SKILLS_DATA_VERSION
    end

    -- Initialize empty bank data if missing.
    if (not AutoMake_BankItemQuantity[AM.storeId]) then
        AutoMake_BankItemQuantity[AM.storeId] = {}
    end

    -- Initialize empty production skills if missing.
    if (not AutoMake_TradeSkillInfo[AM.storeId]) then
        AutoMake_TradeSkillInfo[AM.storeId] = {}
    end
end

-- Utility for displaying a chat message (for debugging).
function AM.Chat(level, text)
    AM.chatCount = AM.chatCount + 1

    -- Text we sent as a message.
    if (level == MESG) then
        if (true) then return end
        text = "AutoMake: "
            .. COLOR_YELLOW .. "MESG(" .. AM.chatCount .. "): " .. COLOR_END
            .. text

    -- Informational (tons and tons of loggin).
    elseif (level == INFO) then
        if (true) then return end
        text = "AutoMake: "
            .. COLOR_GREEN .. "INFO(" .. AM.chatCount .. "): " .. COLOR_END
            .. text

    -- Something is wrong.
    elseif (level == WARN) then
        if (true) then return end
        text = "AutoMake: "
            .. COLOR_ORANGE .. "WARN(" .. AM.chatCount .. "): " .. COLOR_END
            .. text

    -- Some bug in the program.
    elseif (level == BUGS) then
        if (false) then return end
        text = "AutoMake: "
            .. COLOR_RED .. "BUGS(" .. AM.chatCount .. "): " .. COLOR_END
            .. text

    -- "Never happens".
    else
        if (false) then return end
        text = "AutoMake: "
            .. COLOR_RED .. "UNKNOWN{" .. level
            .. "}(" .. AM.chatCount .. "): " .. COLOR_END
            .. text
    end
    DEFAULT_CHAT_FRAME:AddMessage(text)
end

-- Utility for displaying a message.
function AM.Message(text)
    AM.Chat(MESG, text)
    UIErrorsFrame:AddMessage("AutoMake: " .. text,
                             1.0, 0.0, 0.0, 1.0, UIERRORS_HOLD_TIME)
end

-- Iterate on an array in sorted order of keys.
function AM.SortedPairs(array)

    -- Create an ordered list of pairs.
    local orderedKeys = {}
    for key in array do
        table.insert(orderedKeys, key)
    end
    table.sort(orderedKeys)

    -- Create an iterator using the above.
    local index = 0
    return function ()
        index = index + 1
        if not orderedKeys[index] then
            return nil
        else
            return orderedKeys[index], array[orderedKeys[index]]
        end
    end
end

-- Join all keys in an array.
function AM.Join(array, separator)
    local text = ""
    for key in array do
        if (text == "") then
            text = key
        else
            text = text .. separator .. key
        end
    end
    return text
end

-- Verify that a parameter is a boolean and not nil.
function AM.NonNilBool(where, what, value)
    if (value == nil) then
        AM.Chat(BUGS, where .. " " .. what .. " is nil")
    end
end

-- Verify that a parameter is a non-empty string.
function AM.NonEmptyString(where, what, value)
    if (value == nil) then
        AM.Chat(BUGS, where .. " " .. what .. " is nil")
    elseif (value == "") then
        AM.Chat(BUGS, where .. " " .. what .. " is empty")
    end
end

-- Verify that a parameter is a positive integer.
function AM.PositiveInt(where, what, value)
    if (value == nil) then
        AM.Chat(BUGS, where .. " " .. what .. " is nil")
    elseif (value <= 0) then
        AM.Chat(BUGS,
                where .. " " .. what .. "(" .. value .. ") is not positive")
    end
end

-- Verify that a parameter is a non-negative integer,
function AM.NonNegativeInt(where, what, value)
    if (value == nil) then
        AM.Chat(BUGS, where .. " " .. what .. " is nil")
    elseif (value < 0) then
        AM.Chat(BUGS,
                where .. " " .. what .. "(" .. value .. ") is negative")
    end
end

-- Verify that a parameter is a non-nil integer,
function AM.NonNilInt(where, what, value)
    if (value == nil) then
        AM.Chat(BUGS, where .. " " .. what .. " is nil")
    end
end

-- Verify that a parameter is a non-nil table,
function AM.NonNilTable(where, what, value)
    if (value == nil) then
        AM.Chat(BUGS, where .. " " .. what .. " is nil")
    end
end
